/*****************************************************************************
**+------------------------------------------------------------------------+**
**|                                                                        |**
**|                Copyright 2010 Mistral Solutions Pvt Ltd.               |**
**|                                                                        |**
**|                                                                        |**
**|                                                                        |**   
**| This program is free software; you can redistribute it and/or          |**
**| modify it under the terms of the GNU General Public License as         |**
**| published by the Free Software Foundation; either version 2 of         |**
**| the License, or (at your option) any later version.                    |**
**|                                                                        |**
**| This program is distributed in the hope that it will be useful,        |**
**| but WITHOUT ANY WARRANTY; without even the implied warranty of         |**
**| MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the           |**
**| GNU General Public License for more details.                           |**
**|                                                                        |**      
**| You should have received a copy of the GNU General Public License      |**
**| along with this program; if not, write to the Free Software            |**
**| Foundation, Inc., 59 Temple Place, Suite 330, Boston,                  |**
**| MA 02111-1307 USA                                                      |**
**+------------------------------------------------------------------------+**
*****************************************************************************/  

/**
 * \file    eeprom.c
 *
 * \brief   Support routines for I2C EEPROM tests
 *
 * This file contains the support routines for configuring and
 * accessing the I2C based Serial EEPROM.
 *
 * \author  0043
 *
 * \version 01a,13aug,2010 Created
 */

/*
 *====================
 * Includes
 *====================
 */
#include "DM814x_i2c.h"
#include "DM814x_EVM.h"
#include "eeprom.h"

#include "stdio.h"

/*
 *====================
 * Function implementations
 *====================
 */
 /** **************************************************************************
 * \n \brief Routine to initialize the I2C EEPROM
 *
 *  This routine initializes the I2C EEPROM .  
 *  
 * \param EEPROM_I2C_BUS_NUM		[IN]	I2C Bus Instance number
 *
 * \return
 * \n      return SUCCESS for success  - Description
 * 
 */

STATUS EEPROM_init( )
{
    DM814xI2cInit( EEPROM_I2C_BUS_NUM );
    return SUCCESS;
}
/** **************************************************************************
 * \n \brief Routine to read from I2C EEPROM
 *
 * This routine reads one page from EEPROM device through I2C . It takes src
 * as a parameter which gives page to be read along with dst buffer where read 
 * data to be stored and length  of the buffer .
 * 
 *  \param src          [IN]    Page number from where data to be read
 *  \param dst          [IN]    Buffer where read data is stored 
 *  \param  length		[IN]	Length of the buffer
 *
 * \return
 * \n      return SUCCESS for success  - Description
 * \n      return FAILED for error   - Description
 */

STATUS EEPROM_read( UINT32 src, UINT32 dst, UINT32 length )
{
	STATUS u32RetVal = SUCCESS;
    UINT8 addr[2];
    UINT8 *input = ( UINT8* )dst;
    UINT8 u8DataWritten  = 0;

    addr[0] = src >> 8;         // HIGH address
    addr[1] = src & 0xff;       // LOW address

    /* Send 16-bit address */
    u32RetVal = DM814xI2cWite(EEPROM_I2C_BUS_NUM, EEPROM_I2C_ADDR, addr, 2, &u8DataWritten);
       
    if ( FAILED == u32RetVal)
    {
		printf ("EEPROM_read Failed.r\r\n");
        return u32RetVal;
    }

    /* Wait for EEPROM to process */
    DM814x_usecDelay( 1000 );

    /* Read data */
    u32RetVal = DM814xI2cRead( EEPROM_I2C_BUS_NUM, EEPROM_I2C_ADDR, input, length, &u8DataWritten);
    if ( FAILED == u32RetVal)
    {
		printf ("EEPROM_read Failed..\r\n");
        return u32RetVal;
    }

    return u32RetVal;
}
/** ***************************************************************************
 * \n \brief Routine to write I2C EEPROM
 *
 * This routine writes one page of EEPROM.It takes the buffer to be written as a 
 * parameter along with pagenumber and length of the buffer.Both page number and 
 * data are  written to EEPROM together .
 * 
 *
 * \param  src		[IN]	 Buffer to be written
 * \param  dst      [IN]     pagenumber of EEPROM
 * \param  length   [IN]     Length of the buffer
 * 
 * \return
 * \n      return SUCCESS for success  - Description
 * \n      return FAILED for error   - Description
 */

STATUS EEPROM_write( UINT32 src, UINT32 dst, UINT32 length )
{
	STATUS u32RetVal = SUCCESS;
    UINT16 i;
    UINT8 cmd[64 + 2];
    UINT8* psrc8 = ( UINT8* )src;
    UINT8* pdst8 = &cmd[2];
    UINT8 u8DataWritten= 0;

    cmd[0] = dst >> 8;       // HIGH address
    cmd[1] = dst & 0xff;     // LOW address

    /* Fill in data */
    for ( i = 0 ; i < length ; i++ )
        *pdst8++ = *psrc8++;

    /* Send 16-bit address and data */
    u32RetVal = DM814xI2cWite( EEPROM_I2C_BUS_NUM, EEPROM_I2C_ADDR, cmd, (2 + length), &u8DataWritten);
    
    if (FAILED == u32RetVal)
    {
		printf ("EEPROM_write failed.\r\n");
        return u32RetVal;
    }

    /* Wait for EEPROM to process */
    DM814x_usecDelay( 1000 );

    return u32RetVal;
}
